/*
Copyright 2008-2009 Elöd Egyed-Zsigmond, Cyril Laitang
Copyright 2009-2011 Samuel Gesche

This file is part of IPRI News Analyzer.

IPRI News Analyzer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

IPRI News Analyzer is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with IPRI News Analyzer.  If not, see <http://www.gnu.org/licenses/>.
*/

package proc.cluster;

import data.base.Database;
import data.base.connectors.LocalSemanticDatabase;

import data.structures.classification.*;

import data.structures.corpus.ClusteringCorpusGraph;
import data.structures.tagging.GraphLemmaItem;
import data.structures.tagging.LemmaItem;

import proc.text.Horloge;

import java.io.File;
import java.io.FileWriter;

import java.util.Vector;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;

/**
 *
 * @author Utilisateur
 */
public class Experience implements Runnable {

    public final static int VECTEUR_PRODUIT_SCALAIRE = 1;
    public final static int VECTEUR_DISTANCE_EUCLIDIENNE = 2;
    public final static int VECTEUR_DISTANCE_DE_MANHATTAN = 3;
    public final static int VECTEUR_MINORANT_SCALAIRE = 4;
    
    public Experience() {
    }

    private Database base;
    private String titre;
    private String desc;
    private ClusteringCorpusGraph corpus;
    private String statut;
    private boolean enrichArticleBase = false;
    private boolean enrichArticleGen = false;
    private boolean enrichArticleCat = false;
    private boolean enrichArticleSem = false;
    private boolean normalisationTaille = false;
    private Map<Operation, Double> poidsOrigines = new HashMap<Operation, Double>();
    private Map<String, Double> poidsCategories = new HashMap<String, Double>();
    private double poidsCategoriesNonSpecifie = 1.0;
    private int algoCalcul = VECTEUR_PRODUIT_SCALAIRE;

    private Corpus cor = new Corpus();
    private ClusterPart resultat = null;

    public void setBase(Database base){
        this.base = base;
    }

    public void setCorpus(ClusteringCorpusGraph corpus){
        this.corpus = corpus;
    }

    public String getDesc() {
        return desc;
    }

    public void setDesc(String desc) {
        this.desc = desc;
    }

    public String getTitre() {
        return titre;
    }

    public void setTitre(String titre) {
        this.titre = titre;
    }

    public boolean isEnrichArticleCat() {
        return enrichArticleCat;
    }

    public void setEnrichArticleCat(boolean enrichArticleCat) {
        this.enrichArticleCat = enrichArticleCat;
    }

    public boolean isEnrichArticleBase() {
        return enrichArticleBase;
    }

    public void setEnrichArticleBase(boolean enrichArticleBase) {
        this.enrichArticleBase = enrichArticleBase;
    }

    public boolean isEnrichArticleGen() {
        return enrichArticleGen;
    }

    public void setEnrichArticleGen(boolean enrichArticleGen) {
        this.enrichArticleGen = enrichArticleGen;
    }

    public void setPoidsOrigine(Operation origine, double poids){
        poidsOrigines.put(origine, new Double(poids));
    }

    public void annulePoidsCategories(){
        poidsCategoriesNonSpecifie = 0.0;
    }

    public void setPoidsCategorie(String categorie, double poids){
        poidsCategories.put(categorie, new Double(poids));
    }

    public boolean isEnrichArticleSem() {
        return enrichArticleSem;
    }

    public void setEnrichArticleSem(boolean enrichArticleSem) {
        this.enrichArticleSem = enrichArticleSem;
    }

    public boolean isNormalisationTaille(){
        return normalisationTaille;
    }

    public void setNormalisationTaille(boolean normalisationTaille){
        this.normalisationTaille = normalisationTaille;
    }

    public int getAlgoCalcul() {
        return algoCalcul;
    }

    public void setAlgoCalcul(int algoCalcul) {
        this.algoCalcul = algoCalcul;
    }

    public int getNbLemmes() {
        if(corpus==null){
            return 0;
        }

        int nb = 0;
        for (int i = 0; i < cor.getArticles().length; i++) {
            nb += cor.getArticles()[i].getAllLemmes().length;
        }
        return nb;
    }

    private void creeNuage() throws data.base.NoBaseException {
        // * Récupération des articles
        System.out.println(Horloge.getHMS()+"  Articles");

        // Récupération de base : les lemmes des articles, tout simplement
        GraphLemmaItem[] lCor = corpus.getItems();
        for(int i=0; i<lCor.length; i++){
            statut = "Enrichissement article de base ("+(i+1)+"/"+lCor.length+");";
            cor.addArticle(lCor[i]);
        }

        if(isEnrichArticleSem()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par champ sémantique ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getChampSemantique(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CHAMP_SEMANTIQUE);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        if(isEnrichArticleCat()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par catégorisation ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationWiki(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.CATEGORISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }

        if(isEnrichArticleGen()){
            for(int i=0; i<cor.getArticles().length; i++){
                statut = "Enrichissement article par généralisation ("+(i+1)+"/"+cor.getArticles().length+")";
                System.out.println(Horloge.getHMS()+"   Article "+(i+1)+"/"+cor.getArticles().length+" : "+cor.getArticles()[i].getId());
                Vector<LemmePondere> newLemmes = new Vector<LemmePondere>();
                for(int j=0; j<cor.getArticles()[i].getAllLemmes().length; j++){
                    //if(j==0) System.out.println("    Test de l'enrichissement par généralisation");
                    LemmePondere lemme = cor.getArticles()[i].getAllLemmes()[j];
                    System.out.println(Horloge.getHMS()+"    Lemme "+(j+1)+"/"+cor.getArticles()[i].getAllLemmes().length+" : "+lemme.getLemme());
                    //if(j==0) System.out.println("    Premier lemme : "+lemme.getLemme()+" ("+lemme.getCategorie()+")");
                    // enrichissement sémantique de 'lemme'.
                    if(base != null){
                        LocalSemanticDatabase lsdb = new LocalSemanticDatabase(base);
                        Set<LemmaItem> enrichissement0 = lsdb.getGeneralisationOnto(lemme.toLemmaItem());
                        LemmaItem[] enrichissement = new LemmaItem[enrichissement0.size()];
                        enrichissement0.toArray(enrichissement);
                        //if(j==0) System.out.println("    Nombre de parents dans l'ontologie : "+enrichissement.length);
                        for(int a=0; a<enrichissement.length; a++){
                            String nom = enrichissement[a].getLemmaName();
                            int quotite = lemme.getNbOccurences();
                            Vector<Operation> orig = (Vector<Operation>)(lemme.getOrigine().clone());
                            orig.addElement(Operation.GENERALISATION);
                            String categorie = enrichissement[a].getLemmaLex();
                            LemmePondere lp = new LemmePondere(nom, orig, categorie, quotite);
                            newLemmes.add(lp);
                        }
                    }
                }
                for(int j=0; j<newLemmes.size(); j++){
                    cor.getArticles()[i].addLemme(newLemmes.elementAt(j));
                }
            }
        }
    }

    private void creeNuages() throws data.base.NoBaseException {
        //******* CONSTITUTION DES NUAGES **************************************
        System.out.println(Horloge.getHMS()+" Constitution des nuages.");
        if(cor != null){
            creeNuage();
        }

        // * Pondération
        statut = "Pondération en cours";
        System.out.println(Horloge.getHMS()+" Pondération");

        if(!poidsOrigines.isEmpty()){
            Vector<Operation> v = new Vector<Operation>(poidsOrigines.keySet());
            for(int i=0; i<v.size(); i++){
                cor.pondereOrigine(v.elementAt(i), poidsOrigines.get(v.elementAt(i)).doubleValue());
            }
        }
        cor.pondereAllCategories(poidsCategoriesNonSpecifie);
        if(!poidsCategories.isEmpty()){
            Vector<String> v = new Vector<String>(poidsCategories.keySet());
            for(int i=0; i<v.size(); i++){
                cor.pondereCategorie(v.elementAt(i), poidsCategories.get(v.elementAt(i)).doubleValue());
            }
        }
    }

    private String[] creeVecteur(NuagePondere[] nuage){
        Set<String> listeLemmes = new HashSet<String>();
        for(int i=0; i<nuage.length; i++){
            for(int j=0; j<nuage[i].getAllLemmes().length; j++){
                LemmePondere aAjouter = nuage[i].getAllLemmes()[j];
                if(aAjouter.getPoidsCategorie()>0) {
                    listeLemmes.add(aAjouter.getLemme()+":"+aAjouter.getCategorie());
                }
            }
        }
        String[] vecteur = new String[listeLemmes.size()];
        listeLemmes.toArray(vecteur);
        return vecteur;
    }

    private boolean demarree = false;
    private boolean termine = true;
    private long depart;
    private long fin;


    @Override
    public void run() {
        demarree = true;
        resultat = null;
        System.out.println(Horloge.getHMS()+titre+" : expérience lancée.");
        if(!termine){
            System.out.println(Horloge.getHMS()+" Erreur : Précédente expérience en cours.");
            return;
        }
        termine = false;
        // Initialisation
        depart = System.currentTimeMillis();
        if(corpus==null){
            // Pas d'article : pas de confrontation
            termine = true;
            fin = System.currentTimeMillis();
            System.out.println(Horloge.getHMS()+" Erreur : Pas d'article.");
            return;
        }
        if(!isEnrichArticleBase()){
            // Pas d'enrichissement de base : on arrête tout
            // (pour l'instant du moins)
            termine = true;
            fin = System.currentTimeMillis();
            System.out.println(Horloge.getHMS()+" Erreur : Pas d'enrichissement de base.");
            return;
        }

        //******* CONSTITUTION DES NUAGES **************************************
        try {
            creeNuages();
        } catch(data.base.NoBaseException nbe){

        }

        //******* Confrontation des nuages *************************************

        statut = "Clustering en cours";
        NuagePondere[] articles = cor.getArticles();
        String[] vecteur = creeVecteur(articles);
        Vector<ClusterPart> clusters = new Vector<ClusterPart>();
        Vector<double[]> matriceClusters = new Vector<double[]>();

        System.out.println(Horloge.getHMS()+" Clustering : "+articles.length+" articles.");

        // Fabrication de la matrice des vecteurs des articles
        for(int i=0; i<articles.length; i++){
            statut = "Vectorisation initiale en cours (article "+(i+1)+"/"+articles.length+")";
            LemmePondere[] lemmes = articles[i].getAllLemmes();
            double[] vecteurArticle = new double[vecteur.length];
            for(int j=0; j<lemmes.length; j++){
                if(lemmes[j].getPoidsCategorie()>0){
                    String lemme = lemmes[j].getLemme()+":"+lemmes[j].getCategorie();
                    int index = 0;
                    for(int a=0; a<vecteur.length; a++){
                        if(vecteur[a].equals(lemme)){
                            index = a;
                            break;
                        }
                    }
                    vecteurArticle[index] = lemmes[j].getPoidsOrigine()*
                            lemmes[j].getPoidsCategorie()*
                            lemmes[j].getNbOccurences();
                }
            }
            matriceClusters.addElement(vecteurArticle);
            clusters.addElement(articles[i]);
        }

        // Export de la matrice pour tests sur d'autres logiciels
        statut = "Sauvegarde fichier en cours.";
        File f = new File("tableArticles.tsv");
        try{
            FileWriter w = new FileWriter(f);
            String s = "Titre\t";
            for(int i=0; i<vecteur.length; i++){
                s += vecteur[i];
                if(i < vecteur.length-1){
                    s += "\t";
                }
            }
            s += "\r\n";
            w.write(s);
            w.flush();
            for(int j=0; j<articles.length; j++){
                statut = "Sauvegarde fichier en cours (article "+(j+1)+"/"+articles.length+")";
                s = articles[j].getId()+"-"+articles[j].getTitre() + "\t";
                for(int i=0; i<vecteur.length; i++){
                    s += matriceClusters.elementAt(j)[i];
                    if (i < vecteur.length - 1) {
                        s += "\t";
                    }
                }
                s += "\r\n";
                w.write(s);
                w.flush();
            }
        } catch(Exception e){
            e.printStackTrace();
        }

        if (algoCalcul == VECTEUR_PRODUIT_SCALAIRE) {
            while (clusters.size() > 1) {
                statut = "Clustering en cours (encore " + (clusters.size() - 1) + " opérations)";

                // Trouver les deux clusters les plus proches
                double scoreMax = -1 * Double.MAX_VALUE;
                int clusterPart1 = -1;
                int clusterPart2 = -1;
                for (int i = 0; i < clusters.size(); i++) {
                    for (int j = i + 1; j < clusters.size(); j++) {
                        double score = 0.0;
                        for (int a = 0; a < vecteur.length; a++) {
                            score += matriceClusters.elementAt(i)[a] * matriceClusters.elementAt(j)[a];
                        }
                        if (score > scoreMax) {
                            scoreMax = score;
                            clusterPart1 = i;
                            clusterPart2 = j;
                        }
                    }
                }

                // Les fusionner
                Cluster nouveauCluster = new Cluster(
                        new ClusterPart[]{clusters.elementAt(clusterPart1),
                            clusters.elementAt(clusterPart2)}, scoreMax);
                double[] nouveauxPoids = new double[vecteur.length];
                LemmePondere[] lemmes = nouveauCluster.getAllLemmes();
                for (int j = 0; j < lemmes.length; j++) {
                    if (lemmes[j].getPoidsCategorie() > 0) {
                        String lemme = lemmes[j].getLemme() + ":" + lemmes[j].getCategorie();
                        int index = 0;
                        for (int a = 0; a < vecteur.length; a++) {
                            if (vecteur[a].equals(lemme)) {
                                index = a;
                                break;
                            }
                        }
                        nouveauxPoids[index] = lemmes[j].getPoidsOrigine() *
                                lemmes[j].getPoidsCategorie() *
                                lemmes[j].getNbOccurences();
                    }
                }
                ClusterPart partie1 = clusters.elementAt(clusterPart1);
                ClusterPart partie2 = clusters.elementAt(clusterPart2);
                clusters.removeElement(partie1);
                clusters.removeElement(partie2);
                double[] poids1 = matriceClusters.elementAt(clusterPart1);
                double[] poids2 = matriceClusters.elementAt(clusterPart2);
                matriceClusters.removeElement(poids1);
                matriceClusters.removeElement(poids2);
                clusters.addElement(nouveauCluster);
                matriceClusters.addElement(nouveauxPoids);
            }
        } else if(algoCalcul == VECTEUR_DISTANCE_EUCLIDIENNE) {
            while (clusters.size() > 1) {
                statut = "Clustering en cours (encore " + (clusters.size() - 1) + " opérations)";

                // Trouver les deux clusters les plus proches
                double scoreMax = -1 * Double.MAX_VALUE;
                int clusterPart1 = -1;
                int clusterPart2 = -1;
                for (int i = 0; i < clusters.size(); i++) {
                    for (int j = i + 1; j < clusters.size(); j++) {
                        double score = 0.0;
                        for (int a = 0; a < vecteur.length; a++) {
                            // Soustraction car il faut minimiser la distance
                            score -= (matriceClusters.elementAt(i)[a] - matriceClusters.elementAt(j)[a]) *
                                    (matriceClusters.elementAt(i)[a] - matriceClusters.elementAt(j)[a]);
                        }
                        if (score > scoreMax) {
                            scoreMax = score;
                            clusterPart1 = i;
                            clusterPart2 = j;
                        }
                    }
                }

                // Les fusionner
                Cluster nouveauCluster = new Cluster(
                        new ClusterPart[]{clusters.elementAt(clusterPart1),
                            clusters.elementAt(clusterPart2)}, scoreMax);
                double[] nouveauxPoids = new double[vecteur.length];
                LemmePondere[] lemmes = nouveauCluster.getAllLemmes();
                for (int j = 0; j < lemmes.length; j++) {
                    if (lemmes[j].getPoidsCategorie() > 0) {
                        String lemme = lemmes[j].getLemme() + ":" + lemmes[j].getCategorie();
                        int index = 0;
                        for (int a = 0; a < vecteur.length; a++) {
                            if (vecteur[a].equals(lemme)) {
                                index = a;
                                break;
                            }
                        }
                        nouveauxPoids[index] = lemmes[j].getPoidsOrigine() *
                                lemmes[j].getPoidsCategorie() *
                                lemmes[j].getNbOccurences();
                    }
                }
                ClusterPart partie1 = clusters.elementAt(clusterPart1);
                ClusterPart partie2 = clusters.elementAt(clusterPart2);
                clusters.removeElement(partie1);
                clusters.removeElement(partie2);
                double[] poids1 = matriceClusters.elementAt(clusterPart1);
                double[] poids2 = matriceClusters.elementAt(clusterPart2);
                matriceClusters.removeElement(poids1);
                matriceClusters.removeElement(poids2);
                clusters.addElement(nouveauCluster);
                matriceClusters.addElement(nouveauxPoids);
            }
        } else if(algoCalcul == VECTEUR_DISTANCE_DE_MANHATTAN) {
            while (clusters.size() > 1) {
                statut = "Clustering en cours (encore " + (clusters.size() - 1) + " opérations)";

                // Trouver les deux clusters les plus proches
                double scoreMax = -1 * Double.MAX_VALUE;
                int clusterPart1 = -1;
                int clusterPart2 = -1;
                for (int i = 0; i < clusters.size(); i++) {
                    for (int j = i + 1; j < clusters.size(); j++) {
                        double score = 0.0;
                        for (int a = 0; a < vecteur.length; a++) {
                            // Soustraction car il faut minimiser la distance
                            score -= Math.abs(matriceClusters.elementAt(i)[a] - matriceClusters.elementAt(j)[a]);
                        }
                        if (score > scoreMax) {
                            scoreMax = score;
                            clusterPart1 = i;
                            clusterPart2 = j;
                        }
                    }
                }

                // Les fusionner
                Cluster nouveauCluster = new Cluster(
                        new ClusterPart[]{clusters.elementAt(clusterPart1),
                            clusters.elementAt(clusterPart2)}, scoreMax);
                double[] nouveauxPoids = new double[vecteur.length];
                LemmePondere[] lemmes = nouveauCluster.getAllLemmes();
                for (int j = 0; j < lemmes.length; j++) {
                    if (lemmes[j].getPoidsCategorie() > 0) {
                        String lemme = lemmes[j].getLemme() + ":" + lemmes[j].getCategorie();
                        int index = 0;
                        for (int a = 0; a < vecteur.length; a++) {
                            if (vecteur[a].equals(lemme)) {
                                index = a;
                                break;
                            }
                        }
                        nouveauxPoids[index] = lemmes[j].getPoidsOrigine() *
                                lemmes[j].getPoidsCategorie() *
                                lemmes[j].getNbOccurences();
                    }
                }
                ClusterPart partie1 = clusters.elementAt(clusterPart1);
                ClusterPart partie2 = clusters.elementAt(clusterPart2);
                clusters.removeElement(partie1);
                clusters.removeElement(partie2);
                double[] poids1 = matriceClusters.elementAt(clusterPart1);
                double[] poids2 = matriceClusters.elementAt(clusterPart2);
                matriceClusters.removeElement(poids1);
                matriceClusters.removeElement(poids2);
                clusters.addElement(nouveauCluster);
                matriceClusters.addElement(nouveauxPoids);
            }
        } else if(algoCalcul == VECTEUR_MINORANT_SCALAIRE) {
            // Calcul de la matrice des similarités
            // (float car la place occupée est assez grande comme cela)
            float[][] similarites = new float[articles.length][articles.length];
            for(int i=0; i<articles.length; i++){
                similarites[i][i] = Float.MAX_VALUE;
            }
            for(int i=0; i<articles.length; i++){
                statut = "Calcul des similarités en cours (encore " + (articles.length - i - 1) + " opérations)";
                for(int j=0; j<articles.length; j++){
                    double[] vecteuri = matriceClusters.elementAt(i);
                    double[] vecteurj = matriceClusters.elementAt(j);
                    float sim = 0.0f;
                    int divi = 0;
                    int divj = 0;
                    // Somme des poids des lemmes en coocurence (on prend à
                    // chaque fois le poids minimal des deux)
                    for(int a=0; a<vecteuri.length; a++){
                        sim += Math.min(vecteuri[a], vecteurj[a]);
                        divi += vecteuri[a];
                        divj += vecteurj[a];
                    }
                    // Division par le nombre de lemmes du plus petit
                    sim /= Math.min(divi, divj);
                    // On obtient le poids moyen de coocurence
                    similarites[i][j] = sim;
                    similarites[j][i] = sim;
                }
            }

            // Formation d'un grand cluster
            ClusterPart[] liste = new ClusterPart[clusters.size()];
            clusters.toArray(liste);
            Cluster pointDeDepart = new Cluster(liste, 0.0);

            // Division récursive jusqu'à ce que tout soit divisé
            // Peu élégant : les structures sont prévues pour une approche
            // bottom up (on rassemble peu à peu), ici c'est l'inverse
            // (top down : on divise peu à peu)
            //Cluster pointDArrivee = decoupe(pointDeDepart, similarites);


            //clusters.removeAllElements();
            //clusters.addElement(pointDArrivee);
        }

        // Terminé !
        resultat = clusters.elementAt(0);
        System.out.println(Horloge.getHMS()+" Expérience terminée.");

        termine = true;
        fin = System.currentTimeMillis();
    }

    public int getSecondesExecution(){
        if(termine){
            return (int)((fin-depart)/1000);
        } else {
            return (int)((System.currentTimeMillis()-depart)/1000);
        }
    }

    public String getStatut(){
        if(!demarree){
            return "Lancer l'expérience";
        } else if(termine){
            return "Expérience terminée";
        } else {
            return statut;
        }
    }

    public boolean isTermine(){
        return termine;
    }

    public ClusterPart getResultat(){
        return resultat;
    }

}
